package cz.drg.clasificator.setting;

import cz.drg.clasificator.setting.program.Program;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.util.JacksonUtil;
import java.io.File;
import static cz.drg.clasificator.util.OutputHelper.*;
import java.io.IOException;
//import org.eclipse.persistence.jaxb.JAXBContextFactory;

/**
 * Class representing all application settings and how to get access to them.
 * 
 * @author Pavel Jirasek
 */
public class Settings {

    private static final String DEFAULT_PROGRAM_SETTING_DIR = "settings/";
    private static final String DEFAULT_PROGRAM_SETTING = DEFAULT_PROGRAM_SETTING_DIR+"programSettings.xml";
    private static final String DEFAULT_PROGRAM_TEST_SETTING_DIR = "src/test/data/";
    private static final String DEFAULT_PROGRAM_TEST_SETTING = DEFAULT_PROGRAM_TEST_SETTING_DIR+"programTestSettings.xml";
    
    private static ProgramSettings programSettings;
    
    /**
     * Load settings for the application from default location.
     */
    public static void loadSettings(){
        loadProgramSetting(DEFAULT_PROGRAM_SETTING);
    }
    
    /**
     * Load settings for the application from specified location. If the file doesnt exist
     * the default setting will be loaded
     * @param settingPath
     */
    public static void loadSettings(String settingPath){
        File setting = new File(settingPath);
        if(!setting.exists()){
            dualLog(String.format(Constants.ERR_SETTINGS_NOT_FOUND, settingPath));
            settingPath = DEFAULT_PROGRAM_SETTING;
        }
        loadProgramSetting(settingPath);
    }
    
    /**
     * Use this only for testing purposes, this load setting file located in data 
     * test folder.
     */
    public static void loadTestSettings(){
        loadProgramSetting(DEFAULT_PROGRAM_TEST_SETTING);
    }
    
    /**
     * Unmarshal program settings using JAXB. 
     */
    private static void loadProgramSetting(String settingPath){
        
        File setting = new File(settingPath);
        
        if(!setting.exists()){
            dualLog(String.format(Constants.ERR_SETTINGS_NOT_FOUND, settingPath));
            System.exit(0);
        }
        
        Program programSettingXML = null;
        try {
            programSettingXML = JacksonUtil.mapper().readValue(setting, Program.class);
        } 
        catch (IOException ex) {
            dualLog(String.format(Constants.ERR_ILLEGAL_SETTINGS_STRUCTURE, settingPath));
            System.exit(-1);
        }
        
        programSettings = new ProgramSettings(programSettingXML);
        
    }
    
    /**
     * Method for getting program settings. It is needed to load setting first before using 
     * this method. 
     * 
     * @see Settings#loadSettings() 
     * @throws IllegalStateException if the settings are not loaded first
     * @return 
     */
    public static ProgramSettings getProgramSettings(){
        if(programSettings == null){
            throw new IllegalStateException("Cannot fetch the program settings before loading it first!");
        }
        return programSettings;
    }
    
}
